/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.file.remote.mina;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.GeneralSecurityException;
import java.security.PublicKey;

import org.apache.camel.CamelContext;
import org.apache.camel.component.file.GenericFileOperationFailedException;
import org.apache.camel.support.ResourceHelper;
import org.apache.camel.util.IOHelper;
import org.apache.sshd.common.config.keys.OpenSshCertificate;
import org.apache.sshd.common.config.keys.PublicKeyEntry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Utility class for loading OpenSSH certificates from various sources.
 * <p/>
 * Supports loading certificates from:
 * <ul>
 * <li>Byte array containing certificate content</li>
 * <li>File path on filesystem</li>
 * <li>Classpath or file: URI using Camel's ResourceHelper</li>
 * </ul>
 * <p/>
 * Priority order when multiple options are configured: certBytes > certUri > certFile
 * <p/>
 * Certificates must be in OpenSSH certificate format (as generated by ssh-keygen -s). Only USER type certificates are
 * supported for client authentication.
 */
public final class MinaSftpCertificateProvider {

    private static final Logger LOG = LoggerFactory.getLogger(MinaSftpCertificateProvider.class);

    private MinaSftpCertificateProvider() {
        // Utility class
    }

    /**
     * Load an OpenSSH certificate from the given configuration.
     * <p/>
     * Priority order:
     * <ol>
     * <li>Certificate byte array (certBytes)</li>
     * <li>Certificate URI/classpath resource (certUri)</li>
     * <li>Certificate file path (certFile)</li>
     * </ol>
     *
     * @param  config                              the SFTP configuration containing certificate parameters
     * @param  context                             the Camel context for resource loading
     * @return                                     the loaded OpenSSH certificate, or null if no certificate is
     *                                             configured
     * @throws GenericFileOperationFailedException if certificate loading fails
     */
    public static OpenSshCertificate loadCertificate(MinaSftpConfiguration config, CamelContext context)
            throws GenericFileOperationFailedException {

        // 1. Try byte array (highest priority)
        if (config.getCertBytes() != null && config.getCertBytes().length > 0) {
            LOG.debug("Loading certificate from byte array");
            return loadFromBytes(config.getCertBytes(), "certBytes");
        }

        // 2. Try URI/classpath
        if (config.getCertUri() != null && !config.getCertUri().isEmpty()) {
            LOG.debug("Loading certificate from URI: {}", config.getCertUri());
            return loadFromUri(config.getCertUri(), context);
        }

        // 3. Try file path
        if (config.getCertFile() != null && !config.getCertFile().isEmpty()) {
            LOG.debug("Loading certificate from file: {}", config.getCertFile());
            return loadFromFile(config.getCertFile());
        }

        LOG.trace("No certificate configured");
        return null;
    }

    /**
     * Load an OpenSSH certificate from a byte array.
     *
     * @param  certBytes                           the certificate content as bytes
     * @param  resourceName                        a name for logging/error messages
     * @return                                     the loaded OpenSSH certificate
     * @throws GenericFileOperationFailedException if loading fails
     */
    public static OpenSshCertificate loadFromBytes(byte[] certBytes, String resourceName)
            throws GenericFileOperationFailedException {
        try (InputStream is = new ByteArrayInputStream(certBytes)) {
            return loadFromStream(is, resourceName);
        } catch (IOException e) {
            throw new GenericFileOperationFailedException(
                    "Failed to load certificate from bytes: " + e.getMessage(), e);
        }
    }

    /**
     * Load an OpenSSH certificate from a file path.
     *
     * @param  filePath                            the path to the certificate file
     * @return                                     the loaded OpenSSH certificate
     * @throws GenericFileOperationFailedException if loading fails
     */
    public static OpenSshCertificate loadFromFile(String filePath)
            throws GenericFileOperationFailedException {
        Path path = Paths.get(filePath);
        if (!Files.exists(path)) {
            throw new GenericFileOperationFailedException(
                    "Certificate file does not exist: " + filePath);
        }
        if (!Files.isReadable(path)) {
            throw new GenericFileOperationFailedException(
                    "Certificate file is not readable: " + filePath);
        }

        try (InputStream is = Files.newInputStream(path)) {
            return loadFromStream(is, filePath);
        } catch (IOException e) {
            throw new GenericFileOperationFailedException(
                    "Failed to load certificate from file '" + filePath + "': " + e.getMessage(), e);
        }
    }

    /**
     * Load an OpenSSH certificate from a URI using Camel's ResourceHelper.
     *
     * @param  uri                                 the URI (classpath:, file:, etc.)
     * @param  context                             the Camel context for resource resolution
     * @return                                     the loaded OpenSSH certificate
     * @throws GenericFileOperationFailedException if loading fails
     */
    public static OpenSshCertificate loadFromUri(String uri, CamelContext context)
            throws GenericFileOperationFailedException {
        try (InputStream is = ResourceHelper.resolveMandatoryResourceAsInputStream(context, uri)) {
            return loadFromStream(is, uri);
        } catch (IOException e) {
            throw new GenericFileOperationFailedException(
                    "Failed to load certificate from URI '" + uri + "': " + e.getMessage(), e);
        }
    }

    /**
     * Load an OpenSSH certificate from an InputStream.
     *
     * @param  is                                  the input stream containing certificate data
     * @param  resourceName                        a name for logging/error messages
     * @return                                     the loaded OpenSSH certificate
     * @throws GenericFileOperationFailedException if loading fails
     */
    private static OpenSshCertificate loadFromStream(InputStream is, String resourceName)
            throws GenericFileOperationFailedException {
        try {
            String certData = IOHelper.loadText(is).trim();

            // Parse the certificate entry
            PublicKeyEntry certEntry = PublicKeyEntry.parsePublicKeyEntry(certData);
            if (certEntry == null) {
                throw new GenericFileOperationFailedException(
                        "Failed to parse certificate from '" + resourceName + "': no valid data found");
            }

            // Resolve to public key
            PublicKey publicKey = certEntry.resolvePublicKey(null, null, null);

            if (!(publicKey instanceof OpenSshCertificate)) {
                throw new GenericFileOperationFailedException(
                        "Loaded key from '" + resourceName + "' is not an OpenSSH certificate. "
                                                              + "Ensure the file is a *-cert.pub file, not a regular public key.");
            }

            OpenSshCertificate certificate = (OpenSshCertificate) publicKey;

            // Validate certificate type and validity
            validateCertificateType(certificate, resourceName);
            validateCertificateValidity(certificate, resourceName);

            LOG.debug("Successfully loaded OpenSSH USER certificate from {}", resourceName);
            return certificate;

        } catch (IOException | GeneralSecurityException e) {
            throw new GenericFileOperationFailedException(
                    "Failed to load certificate from '" + resourceName + "': " + e.getMessage(), e);
        }
    }

    /**
     * Validate that the certificate is a USER type certificate (for client authentication).
     *
     * @param  certificate                         the certificate to validate
     * @param  resourceName                        a name for error messages
     * @throws GenericFileOperationFailedException if certificate is not USER type
     */
    private static void validateCertificateType(OpenSshCertificate certificate, String resourceName)
            throws GenericFileOperationFailedException {
        OpenSshCertificate.Type certType = certificate.getType();
        if (certType != OpenSshCertificate.Type.USER) {
            throw new GenericFileOperationFailedException(
                    "Certificate from '" + resourceName + "' is not a USER certificate. "
                                                          + "Type: " + certType
                                                          + ". Only USER certificates are supported for client authentication.");
        }
    }

    /**
     * Validate that the certificate is currently valid (not expired, not before valid period).
     * <p/>
     * Delegates to {@link OpenSshCertificate#isValidNow(OpenSshCertificate)} for the actual validation.
     *
     * @param  certificate                         the certificate to validate
     * @param  resourceName                        a name for error messages
     * @throws GenericFileOperationFailedException if certificate is not valid
     */
    private static void validateCertificateValidity(OpenSshCertificate certificate, String resourceName)
            throws GenericFileOperationFailedException {
        // Delegate validity check to MINA SSHD's built-in method
        if (!OpenSshCertificate.isValidNow(certificate)) {
            long validAfter = certificate.getValidAfter();
            long validBefore = certificate.getValidBefore();
            throw new GenericFileOperationFailedException(
                    String.format("Certificate from '%s' is not valid. Valid from %s to %s, current time: %s",
                            resourceName,
                            validAfter > 0 ? java.time.Instant.ofEpochSecond(validAfter) : "epoch",
                            validBefore == OpenSshCertificate.INFINITY
                                    ? "forever"
                                    : java.time.Instant.ofEpochSecond(validBefore),
                            java.time.Instant.now()));
        }

        if (LOG.isTraceEnabled()) {
            long validAfter = certificate.getValidAfter();
            long validBefore = certificate.getValidBefore();
            LOG.trace("Certificate validity check passed: validAfter={}, validBefore={}, now={}",
                    validAfter > 0 ? java.time.Instant.ofEpochSecond(validAfter) : "always",
                    validBefore == OpenSshCertificate.INFINITY
                            ? "forever"
                            : java.time.Instant.ofEpochSecond(validBefore),
                    java.time.Instant.now());
        }
    }
}
