/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.shardingsphere.elasticjob.test.natived.it.staticd;

import org.apache.curator.CuratorZookeeperClient;
import org.apache.curator.retry.ExponentialBackoffRetry;
import org.apache.curator.test.InstanceSpec;
import org.apache.curator.test.TestingServer;
import org.apache.shardingsphere.elasticjob.api.JobConfiguration;
import org.apache.shardingsphere.elasticjob.bootstrap.type.ScheduleJobBootstrap;
import org.apache.shardingsphere.elasticjob.reg.zookeeper.ZookeeperRegistryCenter;
import org.apache.shardingsphere.elasticjob.simple.job.SimpleJob;
import org.awaitility.Awaitility;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.condition.EnabledInNativeImage;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.DynamicPropertyRegistry;
import org.springframework.test.context.DynamicPropertySource;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.request.MockMvcRequestBuilders;
import org.springframework.test.web.servlet.result.MockMvcResultHandlers;
import org.springframework.test.web.servlet.result.MockMvcResultMatchers;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;
import org.springframework.web.context.WebApplicationContext;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Objects;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.jupiter.api.Assertions.assertDoesNotThrow;

/**
 * ElasticJob Spring Boot Starter requires that all Spring Boot Applications be shut down before shutting down Zookeeper Server.
 * That's why this unit test uses {@link DirtiesContext}.
 * Refer to <a href="https://github.com/spring-projects/spring-framework/issues/26196">spring-projects/spring-framework#26196</a> .
 */
@DirtiesContext
@SpringBootTest(webEnvironment = org.springframework.boot.test.context.SpringBootTest.WebEnvironment.RANDOM_PORT)
@EnabledInNativeImage
class SpringBootDTest {
    
    private static TestingServer testingServer;
    
    @Autowired
    private ObjectProvider<ZookeeperRegistryCenter> zookeeperRegistryCenterProvider;
    
    private MockMvc mockMvc;
    
    @DynamicPropertySource
    static void elasticjobProperties(final DynamicPropertyRegistry registry) {
        registry.add("elasticjob.regCenter.serverLists", () -> testingServer.getConnectString());
        registry.add("elasticjob.dump.port", InstanceSpec::getRandomPort);
    }
    
    @BeforeAll
    static void beforeAll() throws Exception {
        testingServer = new TestingServer();
        try (
                CuratorZookeeperClient client = new CuratorZookeeperClient(testingServer.getConnectString(),
                        60 * 1000, 500, null,
                        new ExponentialBackoffRetry(500, 3, 500 * 3))) {
            client.start();
            Awaitility.await().atMost(Duration.ofMillis(500 * 60)).ignoreExceptions().until(client::isConnected);
        }
    }
    
    @AfterAll
    static void afterAll() throws IOException {
        testingServer.close();
    }
    
    @BeforeEach
    void setup(final WebApplicationContext webApplicationContext) {
        this.mockMvc = MockMvcBuilders.webAppContextSetup(webApplicationContext)
                .defaultResponseCharacterEncoding(StandardCharsets.UTF_8)
                .build();
    }
    
    @Test
    public void testOneOffJob() throws Exception {
        String contentAsString = mockMvc.perform(
                MockMvcRequestBuilders.get("/execute/manualScriptJob")
                        .characterEncoding(StandardCharsets.UTF_8))
                .andDo(MockMvcResultHandlers.print())
                .andExpectAll(
                        MockMvcResultMatchers.status().isOk(),
                        MockMvcResultMatchers.content().encoding(StandardCharsets.UTF_8))
                .andReturn()
                .getResponse()
                .getContentAsString();
        assertThat(contentAsString, is("{\"msg\":\"OK\"}"));
    }
    
    @Test
    void testIssue2012() {
        ZookeeperRegistryCenter zookeeperRegistryCenter = zookeeperRegistryCenterProvider.getIfAvailable();
        Objects.requireNonNull(zookeeperRegistryCenter);
        JobConfiguration jobConfig = JobConfiguration.newBuilder("test", 1)
                .jobParameter("")
                .cron("0 30 2 ? * *")
                .overwrite(true)
                .jobShardingStrategyType("ROUND_ROBIN")
                .monitorExecution(true)
                .failover(true)
                .build();
        ScheduleJobBootstrap firstJob = new ScheduleJobBootstrap(zookeeperRegistryCenter,
                (SimpleJob) shardingContext -> System.out.println("test"),
                jobConfig);
        ScheduleJobBootstrap secondJob = new ScheduleJobBootstrap(zookeeperRegistryCenter,
                (SimpleJob) shardingContext -> System.out.println("test"),
                jobConfig);
        assertDoesNotThrow(() -> {
            firstJob.schedule();
            secondJob.schedule();
            firstJob.shutdown();
            secondJob.shutdown();
        });
    }
}
