/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.network;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.InetSocketAddress;
import java.net.MulticastSocket;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.ignite.internal.lang.IgniteInternalException;
import org.apache.ignite.internal.logger.IgniteLogger;
import org.apache.ignite.internal.logger.Loggers;
import org.apache.ignite.internal.network.NodeFinder;
import org.apache.ignite.internal.network.serialization.NetworkAddressesSerializer;
import org.apache.ignite.internal.thread.IgniteThreadFactory;
import org.apache.ignite.internal.thread.ThreadOperation;
import org.apache.ignite.internal.util.IgniteUtils;
import org.apache.ignite.lang.ErrorGroups;
import org.apache.ignite.network.NetworkAddress;
import org.jetbrains.annotations.Nullable;

public class MulticastNodeFinder
implements NodeFinder {
    private static final IgniteLogger LOG = Loggers.forClass(MulticastNodeFinder.class);
    private static final byte[] REQUEST_MESSAGE = "IGNT".getBytes(StandardCharsets.UTF_8);
    private static final int RECEIVE_BUFFER_SIZE = 1024;
    public static final int UNSPECIFIED_TTL = -1;
    public static final int MAX_TTL = 255;
    private static final int REQ_ATTEMPTS = 2;
    private static final int POLLING_TIMEOUT_MILLIS = 100;
    private final InetSocketAddress multicastSocketAddress;
    private final int multicastPort;
    private final int resultWaitMillis;
    private final int ttl;
    private final Set<NetworkAddress> addressesToAdvertise;
    private final ExecutorService listenerThreadPool;
    private final String nodeName;
    private volatile boolean stopped = false;

    public MulticastNodeFinder(String multicastGroup, int multicastPort, int resultWaitMillis, int ttl, String nodeName, Set<NetworkAddress> addressesToAdvertise) {
        this.multicastSocketAddress = new InetSocketAddress(multicastGroup, multicastPort);
        this.multicastPort = multicastPort;
        this.resultWaitMillis = resultWaitMillis;
        this.ttl = ttl;
        this.addressesToAdvertise = addressesToAdvertise;
        this.nodeName = nodeName;
        this.listenerThreadPool = Executors.newSingleThreadExecutor((ThreadFactory)IgniteThreadFactory.create((String)nodeName, (String)"multicast-listener", (IgniteLogger)LOG, (ThreadOperation[])new ThreadOperation[0]));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Collection<NetworkAddress> findNodes() {
        HashSet<NetworkAddress> result = new HashSet<NetworkAddress>();
        List<MulticastSocket> sockets = this.createSockets(0, this.resultWaitMillis, false);
        ExecutorService executor = Executors.newFixedThreadPool(sockets.size(), (ThreadFactory)IgniteThreadFactory.create((String)this.nodeName, (String)"multicast-node-finder", (IgniteLogger)LOG, (ThreadOperation[])new ThreadOperation[0]));
        try {
            List futures = sockets.stream().map(socket -> CompletableFuture.supplyAsync(() -> this.findOnSocket((MulticastSocket)socket), executor)).collect(Collectors.toList());
            for (CompletableFuture future : futures) {
                try {
                    result.addAll((Collection)future.get((long)(this.resultWaitMillis * 2) * 2L, TimeUnit.MILLISECONDS));
                }
                catch (Exception e) {
                    LOG.error("Error while waiting for multicast node finder result", (Throwable)e);
                }
            }
        }
        finally {
            MulticastNodeFinder.closeSockets(sockets);
            IgniteUtils.shutdownAndAwaitTermination((ExecutorService)executor, (long)10L, (TimeUnit)TimeUnit.SECONDS);
        }
        LOG.info("Found addresses: {}", new Object[]{result});
        return result;
    }

    private Collection<NetworkAddress> findOnSocket(MulticastSocket socket) {
        HashSet<NetworkAddress> discovered = new HashSet<NetworkAddress>();
        byte[] responseBuffer = new byte[1024];
        try {
            assert (socket.getSoTimeout() == this.resultWaitMillis);
            for (int i = 0; i < 2; ++i) {
                DatagramPacket requestPacket = new DatagramPacket(REQUEST_MESSAGE, REQUEST_MESSAGE.length, this.multicastSocketAddress);
                socket.send(requestPacket);
                this.waitForResponses(responseBuffer, socket, discovered);
            }
        }
        catch (Exception e) {
            LOG.error("Error during multicast node finding on interface: ", (Throwable)e);
        }
        return discovered;
    }

    private void waitForResponses(byte[] responseBuffer, MulticastSocket socket, Set<NetworkAddress> discovered) throws IOException {
        long endTime = System.currentTimeMillis() + (long)this.resultWaitMillis;
        while (System.currentTimeMillis() < endTime) {
            DatagramPacket responsePacket = new DatagramPacket(responseBuffer, responseBuffer.length);
            try {
                byte[] data = MulticastNodeFinder.receiveDatagramData(socket, responsePacket);
                Set<NetworkAddress> addresses = NetworkAddressesSerializer.deserialize(data);
                if (this.addressesToAdvertise.contains(addresses.iterator().next())) continue;
                discovered.addAll(addresses);
            }
            catch (SocketTimeoutException socketTimeoutException) {}
        }
    }

    private static Collection<NetworkInterface> getEligibleNetworkInterfaces() {
        try {
            return NetworkInterface.networkInterfaces().filter(itf -> {
                try {
                    return itf.isUp() && !itf.isLoopback() && itf.supportsMulticast();
                }
                catch (SocketException e) {
                    LOG.error("Error checking network interface {}", (Throwable)e, new Object[]{itf.getName()});
                    return false;
                }
            }).collect(Collectors.toSet());
        }
        catch (SocketException e) {
            LOG.error("Error getting network interfaces for multicast node finder", (Throwable)e);
            return Set.of();
        }
    }

    public void close() {
        this.stopped = true;
        IgniteUtils.shutdownAndAwaitTermination((ExecutorService)this.listenerThreadPool, (long)10L, (TimeUnit)TimeUnit.SECONDS);
    }

    public void start() {
        List<MulticastSocket> sockets = this.createSockets(this.multicastPort, 100, true);
        if (sockets.isEmpty()) {
            throw new IgniteInternalException(ErrorGroups.Common.INTERNAL_ERR, "No sockets for multicast listener were created.");
        }
        this.listenerThreadPool.submit(() -> {
            byte[] responseData = NetworkAddressesSerializer.serialize(this.addressesToAdvertise);
            byte[] requestBuffer = new byte[REQUEST_MESSAGE.length];
            while (!this.stopped) {
                for (MulticastSocket socket : sockets) {
                    if (socket.isClosed()) continue;
                    try {
                        DatagramPacket requestPacket = new DatagramPacket(requestBuffer, requestBuffer.length);
                        byte[] received = MulticastNodeFinder.receiveDatagramData(socket, requestPacket);
                        if (!Arrays.equals(received, REQUEST_MESSAGE)) {
                            LOG.error("Received unexpected request on multicast socket", new Object[0]);
                            continue;
                        }
                        DatagramPacket responsePacket = new DatagramPacket(responseData, responseData.length, requestPacket.getAddress(), requestPacket.getPort());
                        socket.send(responsePacket);
                    }
                    catch (SocketTimeoutException requestPacket) {
                    }
                    catch (Exception e) {
                        if (!this.stopped) {
                            LOG.error("Error in multicast listener, stopping socket on {}", (Throwable)e);
                        }
                        socket.close();
                    }
                }
            }
            MulticastNodeFinder.closeSockets(sockets);
        });
    }

    private List<MulticastSocket> createSockets(int port, int soTimeout, boolean joinGroup) {
        ArrayList<MulticastSocket> sockets = new ArrayList<MulticastSocket>();
        for (NetworkInterface networkInterface : MulticastNodeFinder.getEligibleNetworkInterfaces()) {
            this.addSocket(sockets, port, networkInterface, soTimeout, joinGroup);
            if (joinGroup) {
                LOG.info("Multicast listener socket created for interface: {}", new Object[]{networkInterface.getDisplayName()});
                continue;
            }
            LOG.info("Multicast node finder socket created for interface: {}", new Object[]{networkInterface.getDisplayName()});
        }
        this.addSocket(sockets, this.multicastPort, null, soTimeout, joinGroup);
        if (sockets.isEmpty()) {
            throw new IgniteInternalException(ErrorGroups.Common.INTERNAL_ERR, "No multicast sockets were created.");
        }
        return sockets;
    }

    private void addSocket(Collection<MulticastSocket> sockets, int port, @Nullable NetworkInterface networkInterface, int soTimeout, boolean joinGroup) {
        try {
            MulticastSocket socket = new MulticastSocket(port);
            socket.setLoopbackMode(false);
            if (networkInterface != null) {
                socket.setNetworkInterface(networkInterface);
            }
            socket.setSoTimeout(soTimeout);
            if (this.ttl != -1) {
                socket.setTimeToLive(this.ttl);
            }
            if (joinGroup) {
                socket.joinGroup(this.multicastSocketAddress, networkInterface);
            }
            sockets.add(socket);
        }
        catch (IOException e) {
            if (networkInterface != null) {
                LOG.error("Failed to create multicast socket for interface {}", (Throwable)e, new Object[]{networkInterface.getName()});
            }
            LOG.error("Failed to create multicast socket for an unbound interface", (Throwable)e);
        }
    }

    private static byte[] receiveDatagramData(MulticastSocket socket, DatagramPacket responsePacket) throws IOException {
        socket.receive(responsePacket);
        return Arrays.copyOfRange(responsePacket.getData(), responsePacket.getOffset(), responsePacket.getOffset() + responsePacket.getLength());
    }

    private static void closeSockets(List<MulticastSocket> sockets) {
        try {
            IgniteUtils.closeAll(sockets);
        }
        catch (Exception e) {
            LOG.error("Could not close multicast listeners", (Throwable)e);
        }
    }
}

